/***************************************************************************************/
/**                          Main window of the FG4 tester.
 *                          (c)2019-2020 Digiteq automotive
 ****************************************************************************************/
#include <vector>
#include <fstream>
#include <sstream>

#include <QString>
#include <QSettings>
#include <QMessageBox>
#include <QFileDialog>
#include <QtGui>
#include <QApplication>
#include <QMessageBox>
#include <QWidget>
#include <QtNetwork>
#include <QScrollBar>

#include "Frm_OutGen.h"

#ifdef _WINDOWS
  #include <Windows.h>
  #include <process.h>
#else
  #include <string.h>
  #include <poll.h> 
  #include <errno.h>  
#endif //_WINDOWS

#include "logfile.h"
#include "logmsg.h"
#include "logQt.h"

#include "xdma.h"
#include "BarAccess.h"

#include "stringa.h"
#include "raster.h"
#include "ras_prot.h"


#define IN1_RESOLUTION_R	0x08
#define IN2_RESOLUTION_R	0x38
#define OUT1_STATUS_R		0x64
#define OUT1_RESOLUTION_RW	0x68
#define OUT2_STATUS_R		0x84
#define OUT2_RESOLUTION_RW	0x88
#define OUT1_CONFIG_RW		0x60
#define OUT2_CONFIG_RW		0x80


extern "C" {
unsigned GetTickCount_ms(void);
}

///////////////////////////////////////////////////////////////////////////////////////////////


class Raster2D_8BitRGBAext: public Raster2D_8BitRGBA
{
public:
	Raster2D_8BitRGBAext(unsigned Size1D, unsigned Size2D, char *Blob);
	~Raster2D_8BitRGBAext();
};


Raster2D_8BitRGBAext::Raster2D_8BitRGBAext(unsigned NewSize1D, unsigned NewSize2D, char *Blob)
{
  if(Blob==NULL)
  {
    Size1D = Size2D = 0;
    return;
  }

  Data2D = (void **)malloc(sizeof(void *)*NewSize2D);
  if(Data2D==NULL)
  {
    Size1D = Size2D = 0;
    return;
  }

  for(unsigned i=0; i<NewSize2D; i++)
  {    
    Data2D[i] = Blob + 4*i*NewSize1D;
  }
  Size1D = NewSize1D;
  Size2D = NewSize2D;
}

Raster2D_8BitRGBAext::~Raster2D_8BitRGBAext()
{
  free(Data2D);
  Data1D = NULL;
  Data2D = NULL;
  Size1D = Size2D = 0;
}


#if QT_VERSION <0x050000
 #define CONN_TYPE	bool
#else
 #define CONN_TYPE	QMetaObject::Connection
#endif


extern LogFile g_TgtLogFile;
extern TgtQt   g_TgtLogQt;
LogFile TgtLogFile2(100000 /*filesize (bytes)*/, 1 /*maxfiles*/);

#define DEVICE_NAME "FG4 Test GUI"



///////////////////////////////////////////////////////////////////////////////////////////////

unsigned int __stdcall FG4_TestGui::FG4_JobThread(void *param)
{
FG4_TestGui *pWindow = (FG4_TestGui*)param;
const HudaqResourceInfo *HRI;

  if(param==NULL) return -1;

  if(pWindow->hDaq==NULL) return -2;
  HRI = HudaqGetDeviceResources(pWindow->hDaq);

  while(!pWindow->SigTerminateThread && pWindow->hDaq!=NULL)
  {
    HudaqWaitForIrq(pWindow->hDaq, 500, 1);			// Wait 100ms for possible IRQ
    pWindow->IrqLoop++;

    DWORD x = GetDword((size_t)HRI->MemResources[1].Base,0x2048);	// Unmasked IRQ requests.
    DWORD chnlIntPendReg = GetDword((size_t)HRI->MemResources[1].Base,0x204C);

    DWORD RegB4 = 0; 
    if(x & 1)
    {
      RegB4 = 1;
      pWindow->Frame0++;
      //pWindow->Frame0ready();		 // poslat signal o pijet obrazku
    }
    if(x & 2)
    {
      RegB4 |= 2;
      pWindow->Frame1++;
      //pWindow->Frame1ready();		 // poslat signal o pijet obrazku
    }
    if(RegB4 & 3)
    {
      StoreDword((size_t)HRI->MemResources[1].Base, 0x2008, RegB4 & 3);	// W1S - Unblock "User Interrupt Enable Mask"
      StoreDword((size_t)HRI->MemResources[0].Base, 0xB4, RegB4 & 3);	// Retrig frame grabbing.      
    }
    
  }

  return 0;
}


FG4_TestGui::FG4_TestGui(QWidget *parent): QDialog(parent), m_ui(new Ui::OutputGen)
{
   m_fileName = "";
   m_IsDirty = false;
   //m_TimerRunning = false;
   hDaq = 0;
   LastBaseAddr = 0xFFFFFFFF;
   FileCounter = 0;
   FrameThread = NULL;
   SigTerminateThread = false;
   IrqLoop = Frame0 = Frame1 = 0;
   oldIrqLoop = oldFrame0 = oldFrame1 = 0;

   m_ui->setupUi(this);

   pImg = new QImage(512, 400, QImage::Format_RGB32);
   for(unsigned x=0; x<256; x++)
     for(unsigned y=0; y<256; y++)
     {
       QRgb value;   
       value = qRgb(rand()%256, rand()%256, rand()%256); // 0xffbd9527
       pImg->setPixel(x, y, value);
     }   
   
   //m_ui->Lbl_Image->setPixmap(QPixmap::fromImage(*pImg));
   //m_ui->Lbl_Image->setVisible(true);

   
   //m_ui->Tbr_main->setVisible(false);

	// Fill some pre-selected values

	// Load (set up) table

	// Update the gui user controls
   update();
   
	// Timer - periodic
   //m_timer2 = new QTimer(this);
   //connect(m_timer2, SIGNAL(timeout()), this, SLOT(slotStatusBar()));
   //m_timer2->setInterval(1000);

   //connect(this, SIGNAL(Frame0ready()), this, SLOT(slotGetFrame()));

   //m_timer = new QTimer(this);
   //connect(m_timer, SIGNAL(timeout()), this, SLOT(slotGetFrame()));
   //m_timer->setInterval(100);

   m_IsAbortionRequested = false;
   
	/* Connect a target to display the logged messages to be used within the GUI */
   CONN_TYPE Result = QObject::connect(&g_TgtLogQt, SIGNAL(logMsgAdded(QString)), this, SLOT(slotLogMsgAdded(QString)));
   if(!Result)
            LogMessageQT(LOG_FATAL, tr("Cannot connect to logger slot.")).Print();

	// Run the full self-check
   QTimer::singleShot(500, this, SLOT(slotTriggered_SelfCheck()));

    //m_ui->centralwidget->setWindowTitle(DEVICE_NAME " Control Desk");
   this->setWindowTitle(DEVICE_NAME " FG4 Test GUI");
   StartupUDP();
}


FG4_TestGui::~FG4_TestGui()
{
  SigTerminateThread = true;
  //if(m_timer) {delete m_timer; m_timer=NULL;}
  //if(m_timer2) {delete m_timer2; m_timer2=NULL;}
  if(pImg) {delete(pImg); pImg=NULL;}

  if(FrameThread!=NULL)
  {
    if(WaitForSingleObject(FrameThread, 5000) == WAIT_TIMEOUT) // Wait for thread to be finished WIN.
    {
      TerminateThread(FrameThread,-1000);	// Kill hanged thread.
    }
    CloseHandle(FrameThread);
    FrameThread = 0;
  }

  if(hDaq != 0)
  {
    HudaqCloseDevice(hDaq);
    hDaq = 0;
  }
  
  if(m_ui) {delete m_ui; m_ui=NULL;}
}


/** Event on close - request of confirmation */
void FG4_TestGui::closeEvent(QCloseEvent *event)
{
  event->accept(); 
}


/** Update window title, used in Save/Save As procedures */
void FG4_TestGui::updateWindowTitle(QString scenarioName)
{
  //setWindowTitle(QString(scenarioName + tr(" - %0, V%1")).arg(VERSIONPRODUCTNAME).arg(VERSIONSTR));
}


/** Key press event */
void FG4_TestGui::keyPressEvent(QKeyEvent *keyEvent)
{
  switch(keyEvent->key())     // "Switch" here in case of future extension to more keys
  {
    case Qt::Key_F10:
        break;
  }
}


/// Load configuration from a given file.
void FG4_TestGui::LoadConfiguration(const QString &conf_file)
{
  std::ifstream infile;
  infile.open(conf_file.toLocal8Bit().constData());

  if(infile.fail())
  {
    LogMessageQT(LOG_ERROR, tr("Cannot open configuration file: '%0'").arg(conf_file)).Print();
    return;
  }  
  
  infile.close();  
}


/** Generally, every slot should be logged. There must be an exception here, otherwise logging
    of this activity would lead to an infinite loop of logging actions. */
void FG4_TestGui::slotLogMsgAdded(QString MSG)
{ 
}


/// Clear table with logs.
void FG4_TestGui::slotClearLog(void)
{  
}


/// Exit application.
void FG4_TestGui::slot_Exit(void)
{
  done(QDialog::Accepted);
}


void FG4_TestGui::slot_Connect(void)
{
DWORD dw;
int DeviceNumber = m_ui->Spin_DevOrder->value();
  if(DeviceNumber<=0) return;

  if(hDaq==0)
  {
    hDaq = HudaqOpenDevice("FG4",DeviceNumber,0);
    if(hDaq==0)
    {
      QMessageBox::critical(this, QString("F demo"), tr("No HUDAQ FG4 device %1 cannot be opened.").arg(DeviceNumber));
      return;
    }
  }

  m_ui->Btn_Connect->setEnabled(false);
  m_ui->Spin_DevOrder->setEnabled(false);

  m_ui->Btn_Disconnect->setEnabled(true);
  m_ui->Chk_Output1->setEnabled(true);
  m_ui->Chk_Output2->setEnabled(true);
  m_ui->Edit_Width1->setEnabled(true);
  m_ui->Edit_Width2->setEnabled(true);
  m_ui->Edit_Height1->setEnabled(true);
  m_ui->Edit_Height2->setEnabled(true);
  m_ui->Cmb_Source1->setEnabled(true);
  m_ui->Cmb_Source2->setEnabled(true);
  m_ui->Btn_Apply->setEnabled(true);

  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);

  dw = GetDword(HRI->MemResources[0].Base, OUT1_STATUS_R);
  m_ui->Chk_Output1->setChecked((dw&1)==1);
  dw = GetDword(HRI->MemResources[0].Base, OUT2_STATUS_R);
  m_ui->Chk_Output2->setChecked((dw&1)==1);
  dw = GetDword(HRI->MemResources[0].Base, OUT1_RESOLUTION_RW);
  m_ui->Edit_Width1->setText(QString::number(dw>>16));
  m_ui->Edit_Height1->setText(QString::number(dw&0xFFFF));  
  dw = GetDword(HRI->MemResources[0].Base, OUT2_RESOLUTION_RW);
  m_ui->Edit_Width2->setText(QString::number(dw>>16));
  m_ui->Edit_Height2->setText(QString::number(dw&0xFFFF));

  return;
}


void FG4_TestGui::slot_Disconnect(void)
{
  if(hDaq!=0)
  {
    HudaqCloseDevice(hDaq);
    hDaq = 0;
  }
  m_ui->Btn_Connect->setEnabled(true);
  m_ui->Spin_DevOrder->setEnabled(true);

  m_ui->Btn_Disconnect->setEnabled(false);
  m_ui->Chk_Output1->setEnabled(false);
  m_ui->Chk_Output2->setEnabled(false);
  m_ui->Edit_Width1->setEnabled(false);
  m_ui->Edit_Width2->setEnabled(false);
  m_ui->Edit_Height1->setEnabled(false);
  m_ui->Edit_Height2->setEnabled(false);
  m_ui->Cmb_Source1->setEnabled(false);
  m_ui->Cmb_Source2->setEnabled(false);
  m_ui->Btn_Apply->setEnabled(false);
}


void FG4_TestGui::slot_Apply(void)
{
QString Str;
unsigned Width, Height;
bool Ok;

  if(hDaq==0)
  {
    return;
  }
  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);

  Str = m_ui->Edit_Width1->text();
  Width = Str.toInt(&Ok);
  if(!Ok)
  {
    QMessageBox::critical(this, QString("F demo"), tr("Invalid value in Width1 \"%1\".").arg(Str));
    return;
  }
  Str = m_ui->Edit_Height1->text();
  Height = Str.toInt(&Ok);
  if(!Ok)
  {
    QMessageBox::critical(this, QString("F demo"), tr("Invalid value in Height1 \"%1\".").arg(Str));
    return;
  }
  StoreDword(HRI->MemResources[0].Base, OUT1_RESOLUTION_RW, (Height&0xFFFF) | (Width<<16));

  Str = m_ui->Edit_Width2->text();
  Width = Str.toInt(&Ok);
  if(!Ok)
  {
    QMessageBox::critical(this, QString("F demo"), tr("Invalid value in Width2 \"%1\".").arg(Str));
    return;
  }
  Str = m_ui->Edit_Height2->text();
  Height = Str.toInt(&Ok);
  if(!Ok)
  {
    QMessageBox::critical(this, QString("F demo"), tr("Invalid value in Height2 \"%1\".").arg(Str));
    return;
  }
  StoreDword(HRI->MemResources[0].Base, OUT2_RESOLUTION_RW, (Height&0xFFFF) | (Width<<16));

  Ok = m_ui->Chk_Output1->checkState() == Qt::CheckState::Checked;
  StoreDword(HRI->MemResources[0].Base, OUT1_CONFIG_RW, Ok?1:0);

  Ok = m_ui->Chk_Output2->checkState() == Qt::CheckState::Checked;
  StoreDword(HRI->MemResources[0].Base, OUT2_CONFIG_RW, Ok?1:0);    
}


void FG4_TestGui::slot_ChangeIndex1(int Idx)
{
DWORD dw;
  if(hDaq==0) return;
  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);

  switch(Idx)
  {
    case 0: dw = GetDword(HRI->MemResources[0].Base, IN1_RESOLUTION_R);
	    m_ui->Edit_Width1->setText(QString::number(dw>>16));
	    m_ui->Edit_Height1->setText(QString::number(dw&0xFFFF));
	    break;
    case 1: dw = GetDword(HRI->MemResources[0].Base, IN2_RESOLUTION_R);
	    m_ui->Edit_Width1->setText(QString::number(dw>>16));
	    m_ui->Edit_Height1->setText(QString::number(dw&0xFFFF));
	    break;
  }
}


void FG4_TestGui::slot_ChangeIndex2(int Idx)
{
DWORD dw;
  if(hDaq==0) return;
  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);

  switch(Idx)
  {
    case 0: dw = GetDword(HRI->MemResources[0].Base, IN1_RESOLUTION_R);
	    m_ui->Edit_Width2->setText(QString::number(dw>>16));
	    m_ui->Edit_Height2->setText(QString::number(dw&0xFFFF));
	    break;
    case 1: dw = GetDword(HRI->MemResources[0].Base, IN2_RESOLUTION_R);
	    m_ui->Edit_Width2->setText(QString::number(dw>>16));
	    m_ui->Edit_Height2->setText(QString::number(dw&0xFFFF));
	    break;
  }
}
