/***************************************************************************************/
/**                            Main window of the FG4 tester.
 *                           (c)2020-2021 Digiteq automotive
 ****************************************************************************************/
#include <QApplication>
#include <QMessageBox>
#include <QDir>
#include <QTranslator>
#include <QLibraryInfo>
#include <QLocale>

#include <signal.h>

#include "Frm_OutGen.h"

#ifdef _WINDOWS
  #include <Windows.h>
  #include "StackWalker.h"
#else
 #define WSACleanup()	/* nothing */
 #define WSAStartup(_x,_y)	(0)	/* return value 0 */
 #define WSADATA int
#endif //_WINDOWS
#include "logfile.h"
#include "logQt.h"


// logging
LogFile g_TgtLogFile(10000000 /*filesize (bytes)*/, 99 /*maxfiles*/); ///< Static instance of file logging target.
TgtQt   g_TgtLogQt;                       ///< Static instance of GUI logging target.
QString g_LogDir("log");
int UDP_Instances = 0;


#ifdef _WIN32
// Simple implementation of an additional output to the console:
class HwSwStackWalker: public StackWalker
{
public:
  HwSwStackWalker(): StackWalker() {}
  HwSwStackWalker(DWORD dwProcessId, HANDLE hProcess): StackWalker(dwProcessId, hProcess) {}
  virtual void OnOutput(LPCSTR szText)
  {
    LogMessage(LOG_FATAL, "CALL STACK %s.", szText).Print();
    // printf(szText);
    StackWalker::OnOutput(szText);
  }
};
#endif


/** Add one reference to UDP ussage.
* @return	<0 - UDP initialisation failed; 1 - UDP reference added. */
int StartupUDP(void)
{
WSADATA wsaData;
  if(UDP_Instances<=0)
  {
    if(WSAStartup(MAKEWORD(2,0), &wsaData)) return -1;
    UDP_Instances = 0;
  }
  UDP_Instances++;
  return 1;
}


/** Release one reference to UDP ussage.
 * @return	0 - UDP is no longer used; 1 - UDP is still used. */
int CloseUDP(void)
{
  UDP_Instances--;
  if(UDP_Instances>0) return 1;
  WSACleanup();
  return 0;
}



/*****************************************************/
/** This is a main signal handler. */
void sig_int(int signo)
{
static int SigSegvEmited = 0;
    // printf("Signal received %d!"); fflush(stdout);

  switch(signo)
  {
    case SIGSEGV:
        if(SigSegvEmited>=1) {abort();break;}
        SigSegvEmited++;
        LogMessage(LOG_FATAL, "Access violation, please report a bug.").Print();
        break;
  }
}


#ifdef _WIN32
LONG Win32FaultHandler(struct _EXCEPTION_POINTERS *ExInfo)
{
  try
  {
    LogMessage(LOG_FATAL, "Access violation at %p, please report a bug.",ExInfo->ExceptionRecord->ExceptionAddress).Print();

    HwSwStackWalker sw;
    sw.ShowCallstack(GetCurrentThread(),ExInfo->ContextRecord);

    g_TgtLogFile.FlushFile();
  }
  catch(...) {}
  return EXCEPTION_EXECUTE_HANDLER;
}
#endif


static void HookSignals(void)
{
#ifdef _WIN32
  SetUnhandledExceptionFilter((LPTOP_LEVEL_EXCEPTION_FILTER) Win32FaultHandler);
#else
  signal(SIGSEGV, sig_int);
#endif
}


// return value: true  = directory for logging exists
//               false = directory for logging cannot be created
bool InitLog(const char *LogPath, MSGCLASS LogLevel)
{
  bool logDirExists = false;

    // initialization of logfile
  g_TgtLogFile.Level = LogLevel;
  g_TgtLogQt.Level = LogLevel;

  if(QDir().exists(QString::fromStdString(LogPath)))
  {
    logDirExists = true;
  }
  else
  {
    if(QDir().mkpath(QString::fromStdString(LogPath)))
    {
      logDirExists = true;
    }
  }
  g_TgtLogFile.SetLogFile(LogPath, "FG4_GUI", "log", 2);
    // TODO: to je stale pravda? if digits=0, an infinite loop appears while trying to delete the old file
    // the number in the filename represents the order of the first log inside
  LogMessage::AttachTarget(&g_TgtLogFile);
  LogMessage::AttachTarget(&g_TgtLogQt);

  LogMessage::SetFlags(LOG_REMOVEEOL);

  return logDirExists;
}


void DO_CRASH(void)
{
  int *i = NULL;
  *i = 0xFFFF;
}


int main(int argc, char *argv[])
{
QString Locale = QLocale::system().name();

    /* Within the installer procedures, this mutex prevents a user from installing new versions
    of this application while the application is still running, and to prevent the user
    from uninstalling a running application. */
 #ifdef _WINDOWS
    /* Nice to have: Would be nice to have this global mutex also set, but here it is not needed and requires testing.
    Can be even extracted into a separate function, but access to logging is required. */
  HANDLE m_glob = CreateMutex(NULL, FALSE, "Global\\FG4TestGUI_Mutex");
  if(m_glob==NULL)	// One global mutex is sufficient enough if it is really created.
  {    
    CreateMutex(NULL, FALSE, "FG4TestGUI_Mutex");  // If not, create only local mutex.
  }
 #endif //_WINDOWS

  HookSignals();
    
  QApplication a(argc, argv);		// init main window
  FG4_TestGui w;

//  bool logDirExists = true; // InitLog(w.m_settings.m_LogDir.replace("\\", "/").toLocal8Bit().constData(), LOG_DEBUG);
  bool logDirExists = InitLog(g_LogDir.replace("\\", "/").toLocal8Bit().constData(), LOG_DEBUG);

        // Detect version of Qt
  LogMessageQT(LOG_INFO, QString("Running on Qt version %0, built against version %1.").arg(qVersion()).arg(QT_VERSION_STR));
  

  for(int i=1; i<argc; i++)
  {
     const char *szToken = argv[i];
     LogMessage(LOG_DEBUG,"Command line argument %s.",szToken).Print();

     if(strcmp(szToken, "--config") == 0 || strcmp(szToken, "-c") == 0)
     {
       if(++i < argc)
       {
         w.LoadConfiguration(argv[i]);
       }       
       continue;
     }
     if(strcmp(szToken, "-L") == 0)
     {      
       if(++i < argc)
       {
	 Locale = argv[i];
       }
       continue;
     }
     if(strcmp(szToken, "--version") == 0)
     {                
       //AboutDialog *dialogAbout = new AboutDialog(&w);
       //dialogAbout->exec();
       //delete dialogAbout;      
       continue;
     }
  }

    // Install Translator
    // Locale can be e.g. "en_US", "en_GB" or "cs_CZ"
  if (Locale.length() > 0)
  {
    QTranslator qtTranslator;
    if (qtTranslator.load("qt_" + Locale, QLibraryInfo::location(QLibraryInfo::TranslationsPath)))
    {
      a.installTranslator(&qtTranslator);
    }
    else if (qtTranslator.load("qt_" + Locale))
    {
      a.installTranslator(&qtTranslator);
    }

    QTranslator myappTranslator;
    if (myappTranslator.load("ControlDesk_" + Locale))
    {
      a.installTranslator(&myappTranslator);
    }
    else
    {
      if (Locale.indexOf("en_") < 0) // Missing translation to any version of English (e.g. US, GB...) is not considered as reason for a warning.
      {
        LogMessage(LOG_WARNING, "Cannot load message translation for locale \"%s\".", Locale.toLocal8Bit().constData()).Print();
      }
    }
  }

  w.show();

  int exec_ret = a.exec();
  CloseUDP();
  return exec_ret;
}
