/* ConsumerFeedGraph.cpp
 *   This demo demonstrates how to connect Direct Show components to a graph.
 *   Code uses smart pointers, there are big troubles with smartpointers and MinGW.
 *   This code connects FG4 renderer to demonstration video source.
 *   (c)2021-2024 Digiteq Automotive  written by Jaroslav Fojtik	*/
#include <Windows.h>
#include <conio.h>

#include <uuids.h>
#include <strmif.h>
#include <control.h>

#include <comdef.h>

#include "FG4iface.h"

#include "stdafx.h"
#include "FrameBuffer.h"
#include "fFrameBuffer.h "


_COM_SMARTPTR_TYPEDEF(IGraphBuilder, __uuidof(IGraphBuilder));
_COM_SMARTPTR_TYPEDEF(IMediaControl, __uuidof(IMediaControl));
_COM_SMARTPTR_TYPEDEF(IAMStreamConfig, __uuidof(IAMStreamConfig));
_COM_SMARTPTR_TYPEDEF(ICaptureGraphBuilder2, __uuidof(ICaptureGraphBuilder2));

_COM_SMARTPTR_TYPEDEF(ICreateDevEnum, __uuidof(ICreateDevEnum));
_COM_SMARTPTR_TYPEDEF(IEnumMoniker, __uuidof(IEnumMoniker));
_COM_SMARTPTR_TYPEDEF(IMoniker, __uuidof(IMoniker));
_COM_SMARTPTR_TYPEDEF(IPin, __uuidof(IPin));
_COM_SMARTPTR_TYPEDEF(IEnumPins, __uuidof(IEnumPins));

_COM_SMARTPTR_TYPEDEF(IFG4KsproxyTransmitConfig, __uuidof(IFG4KsproxyTransmitConfig));
_COM_SMARTPTR_TYPEDEF(IBaseFilter, __uuidof(IBaseFilter));


IMediaControlPtr    graph;			///< containder for camera-grabber stuff
ICaptureGraphBuilder2Ptr capbuilder;		///< Needed for object creation.
CFrameBuffer *MyFrameBuffer = NULL;		///< This object provides fake camera with all signalisation.


unsigned char *pFrameData = NULL;
int FG4_Order = -1;				/// <0 Use standard Windows renderer, >=0 FG4 with given number.

#define VIDEO_WIDTH	1280
#define VIDEO_HEIGHT	640
#define FRAME_RATE	40


////////////////////////////////////////////////////////////////////////////////////////////////////


/// This function returns n'th FG4 renderer.
/// @param[out]	BFilterOut	Smart pointer to FG4 renderer.
/// @param[in]	DEVICE_OUT_ORDER	Device order.
void LocateFg4Renderer(IBaseFilterPtr &BFilterOut, const int DEVICE_OUT_ORDER=0)
{
HRESULT hResult;
    // Locate our capture filter.
    // First, create the device enumerator object

  ICreateDevEnumPtr piCreateDevEnum;
  hResult = piCreateDevEnum.CreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER);

  if(SUCCEEDED(hResult))
  {
        // Get class enumerator for WDM Streaming Capture Devices category
    IEnumMonikerPtr piEnumMoniker;

        // Get class enumerator for WDM Streaming Renderer Devices category
    int Xorder = DEVICE_OUT_ORDER;
    hResult = piCreateDevEnum->CreateClassEnumerator(AM_KSCATEGORY_RENDER, &piEnumMoniker, 0);
    piCreateDevEnum.Detach()->Release();			// no longer needed.

    if(SUCCEEDED(hResult))
        hResult = piEnumMoniker->Reset();

    if(SUCCEEDED(hResult))
    {				// Enumerate KS devices
      ULONG cFetched;
      IMonikerPtr piMoniker;
      while((hResult = piEnumMoniker->Next(1, &piMoniker, &cFetched)) == S_OK)
      {
        IBaseFilterPtr poFilter;

#ifdef _DEBUG
                // Printout devices we find (ignore any errors)
        LPOLESTR pwszMonikerName;
        if(SUCCEEDED(piMoniker->GetDisplayName( NULL, NULL, &pwszMonikerName)))
        {
          printf("Found render device: %S\n", pwszMonikerName);
          CoTaskMemFree(pwszMonikerName);
        }
#endif  // _DEBUG

                // The device we look for is identified by supporting FG4KsproxySampleConfig.
                // Do check this we need to instantiate the filter object and then query it for
                // FG4KsproxySampleConfig.
                // This is not the fastest method.
                // We could rely on filter's name to avoid creating the filter
        hResult = piMoniker->BindToObject(NULL, NULL, __uuidof(poFilter), reinterpret_cast<void**>(&poFilter));
        if(SUCCEEDED(hResult))
        {
	  IFG4KsproxyTransmitConfigPtr poConfigFG4;
          hResult = poFilter->QueryInterface(__uuidof(poConfigFG4), reinterpret_cast<void**>(&poConfigFG4));

          if(FAILED(hResult))
              poFilter.Detach()->Release();
	  else
	  {
	    LPOLESTR pwszMonikerName;
            hResult = piMoniker->GetDisplayName( NULL, NULL, &pwszMonikerName);
            wchar_t *DevName = SUCCEEDED(hResult)?pwszMonikerName:NULL;
	    if(DevName!=NULL)
	    {
	      DevName = wcsstr(pwszMonikerName,L"}\\");
	      if(DevName) DevName+=2;
            }
	    printf("\nDevice transmitter #%d found %S\n", Xorder, (DevName!=NULL)?DevName:L"");
	    if(SUCCEEDED(hResult)) CoTaskMemFree(pwszMonikerName);

	    poConfigFG4.Detach()->Release();
	    if(Xorder > 0)
	    {
	      Xorder--;
	      poFilter.Detach()->Release();
	      hResult = E_FAIL;
            }
            else
            {
	      BFilterOut = poFilter;
	      poFilter.Detach()->Release();
	      break;
            }
          }
        }

        piMoniker.Detach()->Release();
      }
    }
    piEnumMoniker.Detach()->Release();
  }
}


////////////////////////////////////////////////////////////////////////////////////////////////////


/// This function returns standard Windows renderer named display.
/// @param[out]	BFilterOut	Smart pointer to system's renderer.
void LocateClassicRenderer(IBaseFilterPtr &BFilterOut)
{
HRESULT hResult;
    // Locate our capture filter.
    // First, create the device enumerator object

  ICreateDevEnumPtr piCreateDevEnum;
  hResult = piCreateDevEnum.CreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER);

  if(SUCCEEDED(hResult))
  {
        // Get class enumerator for WDM Streaming Capture Devices category
    IEnumMonikerPtr piEnumMoniker;

        // Get class enumerator for Display sink object
    hResult = piCreateDevEnum->CreateClassEnumerator(CLSID_LegacyAmFilterCategory, &piEnumMoniker, 0);
    piCreateDevEnum.Detach()->Release();			// no longer needed.

    if(SUCCEEDED(hResult))
        hResult = piEnumMoniker->Reset();

    if(SUCCEEDED(hResult))
    {				// Enumerate KS devices
      ULONG cFetched;
      IMonikerPtr piMoniker;
      while((hResult = piEnumMoniker->Next(1, &piMoniker, &cFetched)) == S_OK)
      {
        IBaseFilterPtr poFilter;

                // Printout devices we find (ignore any errors)
        LPOLESTR pwszMonikerName;
        if(SUCCEEDED(piMoniker->GetDisplayName( NULL, NULL, &pwszMonikerName)))
        {
          printf("Found render device: %S\n", pwszMonikerName);

          if(!wcscmp(pwszMonikerName,L"@device:sw:{083863F1-70DE-11D0-BD40-00A0C911CE86}\\{6BC1CFFA-8FC1-4261-AC22-CFB4CC38DB50}"))
          {
            hResult = piMoniker->BindToObject(NULL, NULL, __uuidof(poFilter), reinterpret_cast<void**>(&poFilter));
            if(SUCCEEDED(hResult))
            {
              printf("\nDevice transmitter found %S\n", (pwszMonikerName!=NULL)?pwszMonikerName:L"");

              BFilterOut = poFilter;
	      poFilter.Detach()->Release();
              CoTaskMemFree(pwszMonikerName);
	      break;
            }
          }
          CoTaskMemFree(pwszMonikerName);
        }
        piMoniker.Detach()->Release();
      }
    }
    piEnumMoniker.Detach()->Release();
  }
}


////////////////////////////////////////////////////////////////////////////////////////////////////


/// Display hotkeys.
void Help(void)
{
  printf("\nD	Display status");
  printf("\nF	Feed frames to video buffer");
  printf("\nH	Show help");
  printf("\nP	Pause graph");
  printf("\nR	Run graph");
  printf("\nI	Stop graph - sometimes hangs, known problem");
  printf("\nS	Stop graph - should be safe");
  printf("\nX	Exit");
  printf("\n1	Stopping renderer only");
  printf("\n2	Stopping camera only");
}


/// Display command line arguments.
void ShowHelp(void)
{
  printf("Program arguments:\n"
	"  -Help               Display this help\n"
	"  -FG4:number         Use n'th FG4 renderer.");
}


////////////////////////////////////////////////////////////////////////////////////////////////////


/// This function creates some demonstrative video. It pushes video generated into
/// a FrameBuffer object using its newly created method.
void FeedFrameToVideo(void)
{
  unsigned PixPx = (rand() % VIDEO_WIDTH);
  unsigned PixPy = (rand() % VIDEO_HEIGHT);
  unsigned char PrintColor[4] = {rand()%256, rand()%256, rand()%256, rand()%256};

	// Mockup some frame.
  for(int i=PixPx; i<PixPx+20; i++)
    for(int j=PixPy; j<PixPy+20; j++)
    {
      memcpy(pFrameData + 4*VIDEO_WIDTH*(j%VIDEO_HEIGHT)+4*(i%VIDEO_WIDTH), PrintColor, 4);
    }
	// Push videoframe generated to DShow graph.
  if(SUCCEEDED(MyFrameBuffer->FeedExternalBuffer(pFrameData, VIDEO_WIDTH*VIDEO_HEIGHT*4)))
    putchar('.');
  else
    putchar('!');
}


////////////////////////////////////////////////////////////////////////////////////////////////////

/** This function allows user to handle all state transitions.
  Both camera and renderer must be commanded to intended state.
  States are intended to be handled within GUI, this function is here 
  for hemonstration only.
 @param[in]	Camera	Pointer to COM camera object.
 @param[in]	Renderer	Pointer to COM renderer object. */
void HandleGraph(IBaseFilter *Camera, IBaseFilter *Renderer)
{
HRESULT hr;
bool FrameSending = false;
unsigned LastFrame = GetTickCount_ms();
  char key = 0;
  Help();
  while(key!='X')
  {
    if(!kbhit())
    {
      MSG msg;
      while(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))   //Pump for Windows messages, any GUI app has its own pump.
      {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
        if(msg.message == WM_CLOSE) {key='X';break;}
      }
      if(msg.message == WM_CLOSE)
        key='X';
      else
      {
        const unsigned ActTick = GetTickCount_ms();
        if(FrameSending)
        {
          if((ActTick-LastFrame)>FRAME_RATE)
          {
            LastFrame = ActTick;
            FeedFrameToVideo();
          }
          continue;
        }

        Sleep(10);
        continue;
      }
    }
    key = toupper(getch());

	// Displaying state is placed here only for debugging purpose.
    if(key=='D')
    {
      FILTER_STATE StateCR;
      printf("\nPressed D ... Stopping graph (reverse order)");
      hr = Camera->GetState(0, &StateCR);
      if(FAILED(hr))
        printf("\nCannot obtain camera status, error %Xh", (unsigned)hr);
      else
        printf("\nCamera status is %Xh", (unsigned)StateCR);
      hr = Renderer->GetState(0, &StateCR);
      if(FAILED(hr))
        printf("\nCannot obtain renderer status, error %Xh", (unsigned)hr);
      else
        printf("\nRenderer status is %Xh", (unsigned)StateCR);
    }

    if(key=='H')
    {
      Help();
    }

	// Stopping renderer first sometimes causes DShow to hang from unknown
	// reason. It is here only for demonstration & debug purposes.
	// Do not do this in application.
    if(key=='I')
    {
      printf("\nPressed I ... Stopping graph (reverse order)");
      hr = Renderer->Stop();
      if(FAILED(hr))
          printf("\nCannot stop renderrer, error %Xh", (unsigned)hr);
      else
          printf("; renderer stopped");
      hr = Camera->Stop();
      if(FAILED(hr))
          printf("\nCannot stop camera, error %Xh", (unsigned)hr);
      else
          printf("; camera stopped");
    }

    if(key=='1')
    {
      printf("\nPressed 1 ... Stopping renderer only");
      hr = Renderer->Stop();
      if(FAILED(hr))
          printf("\nCannot stop renderrer, error %Xh", (unsigned)hr);
      else
          printf("; renderer stopped");
    }
    if(key=='2')
    {
      printf("\nPressed 2 ... Stopping camera only");
      hr = Camera->Stop();
      if(FAILED(hr))
          printf("\nCannot stop camera, error %Xh", (unsigned)hr);
      else
          printf("; camera stopped");
    }

	// This demonstrates how to correctly stop renderer and
	// camera.
    if(key=='S')
    {
      printf("\nPressed S ... Safe/Reverse stopping graph");
      hr = Camera->Stop();
      if(FAILED(hr))
          printf("\nCannot stop camera, error %Xh", (unsigned)hr);
      else
          printf("; camera stopped");
      hr = Renderer->Stop();
      if(FAILED(hr))
          printf("\nCannot stop renderrer, error %Xh", (unsigned)hr);
      else
          printf("; renderer stopped");
    }

	// Switch DShow graph to a pause state.
    if(key=='P')
    {
      printf("\nPressed P ... Pausing graph");
      hr = Renderer->Pause();
      if(FAILED(hr))
         printf("\nCannot pause renderrer, error %Xh", (unsigned)hr);
      hr = Camera->Pause();
      if(FAILED(hr))
         printf("\nCannot pause camera, error %Xh", (unsigned)hr);
    }

	// Start DShow graph.
    if(key=='R')
    {
      printf("\nPressed R ... Starting graph");
      hr = Renderer->Run(0);
      if(FAILED(hr))
         printf("\nCannot start renderer, error %Xh", (unsigned)hr);
      hr = Camera->Run(0);		// This causes internal thread to be started.
      if(FAILED(hr))
         printf("\nCannot start camera, error %Xh", (unsigned)hr);
    }

    if(key=='F')
    {
      FrameSending = !FrameSending;
      printf("\nPressed F ... Sending frames %s", FrameSending?"True":"False");
      LastFrame = GetTickCount_ms();
    }
  }

	// Please note that leaving running graph causes application to hang.
	// Possible application hang is allowed for demonstrative purposes.
  printf("\nPressed X ... Application exit");
}


////////////////////////////////////////////////////////////////////////////////////////////////////

int UseGraph(void)
{
HRESULT hr;

//////////////Obtain both camera and rendered objects//////////////////////

  IBaseFilterPtr renderer_base;

  if(FG4_Order<0)
    LocateClassicRenderer(renderer_base);	// Use this renderer if you do not have FG4 inserted.
  else
    LocateFg4Renderer(renderer_base, FG4_Order); // Get n'th FG4 renderer.


  if(renderer_base == NULL)
  {
     printf("\nCannot find FG4 renderer");
     return -3;
  }

/////////////////////Build direct show graph///////////////////////////////

	// create the filter graph and necessary interfaces
  hr = graph.CreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC);
  if(FAILED(hr) || graph==NULL)
  {
    printf("\nFailed to create filter graph, error %Xh", (unsigned)hr);
    return -4;
  }
  IGraphBuilderPtr graphbuilder(graph);
  if(graphbuilder==NULL)
  {
    printf("\nFailed to get GraphBuilder interface for the filter graph");
    return -5;
  }

	// create the capture graph builder
  hr = capbuilder.CreateInstance(CLSID_CaptureGraphBuilder2, NULL, CLSCTX_INPROC);
  if(FAILED(hr) || capbuilder==NULL)
  {
    printf("\nFailed to create capture graph builder, error %Xh", (unsigned)hr);
    return -6;
  }

	// attach the filter graph to the capture graph builder
  hr = capbuilder->SetFiltergraph(graphbuilder);
  if(FAILED(hr))
  {
    printf("\nFailed to attach filter graph to the capture graph builder, error %Xh", (unsigned)hr);
    return -7;
  }

	// Add the camera filter to the filter graph
  hr = graphbuilder->AddFilter(MyFrameBuffer, L"Camera");
  if(FAILED(hr))
  {
    printf("\nFailed to add camera filter to filter graph, error %Xh", (unsigned)hr);
    return -8;
  }

	// Add the camera filter to the filter graph
  hr = graphbuilder->AddFilter(renderer_base, L"Renderer");
  if(FAILED(hr))
  {
    printf("\nFailed to add renderer filter to filter graph, error %Xh", (unsigned)hr);
    return -9;
  }

	// Now we need to handle PINs.
  IPinPtr CamPIN;
  IPinPtr RenderPIN;
  IEnumPinsPtr pEnum;

	// Obtain pin enumerator from a camera
  hr = MyFrameBuffer->EnumPins(&pEnum);
  hr = pEnum->Next(1,&CamPIN,NULL);
  if(FAILED(hr))
  {
    printf("\nCannot obtain output pin from camera, error %Xh", (unsigned)hr);
    return -10;
  }

  pEnum.Detach()->Release(); // enumerator is no longer needed.

	// Obtain pin enumerator from a renderer
  hr = renderer_base->EnumPins(&pEnum);
  hr = pEnum->Next(1,&RenderPIN,NULL);
  if(FAILED(hr))
  {
    printf("\nCannot obtain output pin from renderer, error %Xh", (unsigned)hr);
    return -11;
  }

  pEnum.Detach()->Release();

	// Interconnect both pins
  hr = graphbuilder->Connect(CamPIN, RenderPIN);
  if(FAILED(hr))
  {
    printf("\nCannot connect renderer with grabber, error %Xh.", (unsigned)hr);
    //hr = graphbuilder->Render(CamPIN);
    if(FAILED(hr)) return -12;
  }


///////////////////////////////////////////////////////

	// Graph is finally constructed and we can Start/Stop/Pause devices.
  HandleGraph(MyFrameBuffer, renderer_base);

return 0;
}


int main(int argc, char **argv)
{
HRESULT hr;

  printf("<<FeedGraph>> (c) 2023-2024 Digiteq Automotive\n");
  for(int i=1; i<argc; i++)
  {
    if(strcmp("-help",argv[i])==0 || strcmp("/help",argv[i]) == 0)
    {
      ShowHelp();
      return -1;
    }
    if(strncmp("-FG4:",argv[i],5)==0 || strncmp("/FG4:",argv[i],5) == 0)
    {
      FG4_Order = atoi(argv[i]+5);
      continue;
    }
  }

  hr = CoInitializeEx(NULL, COINIT_MULTITHREADED|COINIT_DISABLE_OLE1DDE);
  if(FAILED(hr))
  {
    printf("\nCoInitializeEx failed, error %Xh", (unsigned)hr);
    return -1;
  }

	// Camera is built in object that is created through NEW C++ command.
  MyFrameBuffer = new CFrameBuffer(NULL, &hr);		// Here you can create your own video source.
  MyFrameBuffer->AddRef();				// Add one reference.
  MyFrameBuffer->SetWidth(VIDEO_WIDTH);			// Set a proper resolution
  MyFrameBuffer->SetHeight(VIDEO_HEIGHT);		//   resolution must be set *BEFORE* an object is attached into graph.
  MyFrameBuffer->SetRepeatTime_ms(FRAME_RATE);		// Notify about expected framerate.
  MyFrameBuffer->SetBallSize(60);

  pFrameData = (unsigned char*)calloc(VIDEO_WIDTH * VIDEO_HEIGHT, 4);

  int ret = UseGraph();		// Creating a graph and demonstrate functional graph.

	// Application exit.
  if(!MyFrameBuffer->IsStopped())
      MyFrameBuffer->Stop();				// Ensure that a fake camera is in Stop state. */
  if(graph!=NULL)
     graph.Detach()->Release();

  if(capbuilder!=NULL)
    capbuilder.Detach()->Release();

  if(MyFrameBuffer)			// Destroy framebuffer. delete() must not be used.
  {
    MyFrameBuffer->Release();
    MyFrameBuffer = NULL;
  }

  if(pFrameData)			// Free external simulBuffer 
  {
    free(pFrameData);
    pFrameData = NULL;
  }

  CoUninitialize();
return ret;
}
